/** @file   eng2d_exception.h
 * @brief   Declares the exception classes for eng2d - library.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_ENG2DEXCEPTION_H
#define H_ENG2D_ENG2DEXCEPTION_H

#include <string>
#include "eng2d_dll.h"

namespace eng2d {

/** @class  xBaseException
 * @brief   The baseclass for all the exception classes in eng2d - library.
 * @author  Tomi Lamminsaari
 *
 */
class DLLIMPORT xBaseException
{
public:
  ///
  /// Members
  /// =======
  
  /** Name of the component that caused this exception. */
  std::string iClassName;
  
  /** Name of the method that caused this exception. */
  std::string iMethodName;
  
  /** The message this exception object carries. */
  std::string iMessage;
  
  

  ///
  /// Constants, datatype and static members
  /// ======================================


  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Default constructor. No message.
   */
  xBaseException();
  
  
	/** Constructor.
   * @param     aMessage          The message you want to attach this object.
   */
	explicit xBaseException( const std::string& aMessage );
  
  
  /** Constructor.
   * @param     aClassName        Name of the class that created this.
   * @param     aMethodName       Name of the method where this was created
   * @param     aMessage          The message.
   */
  xBaseException( const std::string& aClassName,
                  const std::string& aMethodName,
                  const std::string& aMessage );


	/** Destructor
   */
	virtual ~xBaseException();
  

	/** Copy constructor.
   * @param     aObj              Reference to another BaseException
   */
  xBaseException( const xBaseException& aObj );
  

	/** Assignment operator
   * @param     aObj              Reference to another BaseException
   * @return    Reference to us.
   */
  xBaseException& operator = ( const xBaseException& aObj );



  ///
  /// Methods
  /// =======



  ///
  /// Getter methods
  /// ==============

};




/** @class  xOutOfBounds
 * @brief   Thrown, when indexing outside the vector bounds.
 * @author  Tomi Lamminsaari
 *
 * The eng2d-libary throws these exceptions when accessing outside an
 * array or a vector structures.
 */
class DLLIMPORT xOutOfBounds : public xBaseException
{
public:

  /** Constructor.
   */
  xOutOfBounds();
  
  /** Constructor. The message will be "out of bounds"
   * @param     aClassName        Name of the class or component who throws
   *                              this.
   * @param     aMethodName       Name of the method
   */
  xOutOfBounds( const std::string& aClassName,
                const std::string& aMethodName );
  
  /** Copy constructor.
   * @param     aObj              Reference to another xOutOfBounds
   */
  xOutOfBounds( const xOutOfBounds& aObj );
  
  /** Destructor.
   */
  virtual ~xOutOfBounds();
  
  /** An assignment operator.
   * @param     aObj              Reference to anothe xOutOfBounds
   * @return    Reference to us
   */
  xOutOfBounds& operator = ( const xOutOfBounds& aObj );
};



/** @class  xLogicError
 * @brief   An exception class for logic errors.
 * @author  Tomi Lamminsaari
 *
 * Exceptions of this type are thrown when some kind of logic errors
 * happen. For example attempt to create a polygon with less than 3
 * vertices throws this exception.
 */
class DLLIMPORT xLogicError : public xBaseException
{
public:
  /** Constructor.
   */
  xLogicError();
  
  /** Constructor.
   * @param     aClassName        Name of the class who creates this.
   * @param     aMethodName       Name of the method where this is needed.
   * @param     aMessage          The message.
   */
  xLogicError( const std::string& aClassName,
               const std::string& aMethodName,
               const std::string& aMessage );
               
  /** Copy constructor.
   */
  xLogicError( const xLogicError& aObj );
  
  /** Destructor.
   */
  virtual ~xLogicError();
  
  /** An assignment operator
   */
  xLogicError& operator = ( const xLogicError& aObj );
};



/** @class  xRunTimeError
 * @brief   An exception class for runtime errors.
 * @author  Tomi Lamminsaari
 *
 * Runtime errors are reported with exceptions of this type. For
 * instance out ot memory situations throw these exceptions.
 */
class DLLIMPORT xRuntimeError : public xBaseException
{
public:
  /** Constructor.
   */
  xRuntimeError();
  
  /** Constructor.
   * @param     aClassName        Name of the class who creates this.
   * @param     aMethodName       Name of the method where this is needed.
   * @param     aMessage          The message
   */
  xRuntimeError( const std::string& aClassName,
                 const std::string& aMethodName,
                 const std::string& aMessage );

  /** Copy constructor.
   */
  xRuntimeError( const xRuntimeError& aObj );
  
  /** Destructor.
   */
  virtual ~xRuntimeError();
  
  /** An assignment operator
   */
  xRuntimeError& operator = ( const xRuntimeError& aObj );
};



/** @class  xIoError
 * @brief   An exception class for io-errors.
 * @author  Tomi Lamminsaari
 *
 * The io-errors generate an exception of this type.
 */
class DLLIMPORT xIoError : public xBaseException
{
public:
  /** Constructor.
   */
  xIoError();
  
  /** Constructor.
   * @param     aClassName        Name of the class who creates this.
   * @param     aMethodName       Name of the method.
   * @param     aMessage          The message
   */
  xIoError( const std::string& aClassName,
            const std::string& aMethodName,
            const std::string& aMessage );

  /** Copy constructor
   */
  xIoError( const xIoError& aObj );
  
  /** Destructor.
   */
  virtual ~xIoError();
  
  /** An assignment operator
   */
  xIoError& operator= ( const xIoError& aObj );
};

};  // end of namespace

#endif
